/* Copyright (C) 2015-2018 RealVNC Ltd. All Rights Reserved.
 */

#include "ConditionAttribute.h"

#include <vnccommon/StringUtils.h>

#include <errno.h>
#include <string>
#include <cstring>
#include <stdexcept>


using namespace vnccommon;

ConditionAttribute::ConditionAttribute()
    : mIsMonotonicClockAvailable(false)
{
# ifdef NON_STANDARD_PTHREAD
    mIsMonotonicClockAvailable = true;
    return;
# else
    const int attr_result = pthread_condattr_init(&mCondAttr);
    if (attr_result != 0)
    {
        std::string message = vnccommon::StringUtils::format(
                "pthread_condattr_init failed. %s", strerror(attr_result));
        throw std::runtime_error(message.c_str());
    }

#   ifndef __APPLE__
    // Apple does not have pthread_condattr_setclock, so monotonic
    // waiting is not available for that platform.
    const int clock_result = pthread_condattr_setclock(&mCondAttr, CLOCK_MONOTONIC);
    if (clock_result == 0)
    {
        // Successfully set the attribute
        // Attempt to get a monotonic timestamp to make sure the monotonic clock
        // works
        struct timespec time;
        if (clock_gettime(CLOCK_MONOTONIC, &time) == 0) {
            mIsMonotonicClockAvailable = true;
        }
    }
#   endif

#endif
}

ConditionAttribute::~ConditionAttribute()
{
#ifndef NON_STANDARD_PTHREAD
    pthread_condattr_destroy(&mCondAttr);
#endif
}

pthread_condattr_t* ConditionAttribute::pthreadCondAttr()
{
    return mIsMonotonicClockAvailable ? &mCondAttr : NULL;
}

bool ConditionAttribute::isMonotonicClockAvailable()
{
    return mIsMonotonicClockAvailable;
}
